/*
 * Copyright (c) 1999-2001 Lutris Technologies, Inc. All Rights
 * Reserved.
 * 
 * This source code file is distributed by Lutris Technologies, Inc. for
 * use only by licensed users of product(s) that include this source
 * file. Use of this source file or the software that uses it is covered
 * by the terms and conditions of the Lutris Enhydra Development License
 * Agreement included with this product.
 * 
 * This Software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
 * ANY KIND, either express or implied. See the License for the specific terms
 * governing rights and limitations under the License.
 * 
 * Contributor(s):
 * 
 * $Id: Element.java,v 1.1.1.1 2001/01/05 05:11:17 sese0235 Exp $
 */

package de.kxml.kdom;

import java.io.*;
import java.util.*;
import de.kxml.*;
import de.kxml.io.*;
import de.kxml.parser.*;

/**
 * Abstract base class for elements. In order to create an element,
 * please use Document.createElement. The right place to add user
 * defined initialization code is probably setParent.
 */
public abstract class Element {

    /**
     * adds a child Node of the given type at the given index to this
     * element. The child's parent must match the current element!!
     */
    public abstract void addChild(int index, int type, Object child);

    /**
     * convenience method for addChild (getChildCount (), child)
     */
    public void addChild(int type, Object child) {
	addChild(getChildCount(), type, child);
    } 

    /**
     * removes all children and attributes
     */
    public void clear() {
	setAttributes(new Vector());

	for (int i = getChildCount() - 1; i >= 0; i--) {
	    removeChild(i);
	} 
    } 

    /**
     * returns the attribute with the given index
     */
    public abstract Attribute getAttribute(int index);

    /**
     * convenience method for getAttribute ("", name)
     */
    public Attribute getAttribute(String name) {
	return getAttribute(Xml.NO_NAMESPACE, name);
    } 

    /**
     * returns the attribute with the given namespace and name
     */
    public Attribute getAttribute(String namespace, String name) {
	int len = getAttributeCount();

	if (namespace == null) {
	    namespace = Xml.NO_NAMESPACE;
	} 

	for (int i = 0; i < len; i++) {
	    Attribute attr = getAttribute(i);

	    if (namespace.equals(attr.getNamespace()) 
		    && name.equals(attr.getName())) {
		return attr;
	    } 
	} 

	return null;
    } 

    /**
     * returns a Vector containing the attributes
     */
    public abstract Vector getAttributes();

    /**
     * returns the number of attributes
     */
    public abstract int getAttributeCount();

    /**
     * returns the child node at the given index.  For child
     * elements, an Element object is returned. For all other child
     * types, a String is returned.
     */
    public abstract Object getChild(int index);

    /**
     * returns the number of child nodes
     */
    public abstract int getChildCount();

    /**
     * returns the element at the given index. If the node at the
     * given index is a text node, null is returned
     */
    public Element getElement(int index) {
	Object child = getChild(index);

	return (child instanceof Element) ? (Element) child : null;
    } 

    /**
     * returns the (local) name of the element
     */
    public abstract String getName();

    /**
     * returns the namespace of the element
     */
    public abstract String getNamespace();

    /**
     * returns the owner document of this element or null if not
     * attached to a document
     */
    public abstract PrefixMap getPrefixMap();

    /**
     * Method declaration
     *
     *
     * @param map
     *
     * @see
     */
    public abstract void setPrefixMap(PrefixMap map);

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @see
     */
    public abstract Document getDocument();

    /**
     * returns the parent element. For root or free elements, null is
     * returned.
     */
    public abstract Element getParentElement();

    /**
     * returns the text content if the element has text-only
     * content. Throws an exception for mixed content
     */
    public String getText() {
	StringBuffer buf = new StringBuffer();
	int	     len = getChildCount();

	for (int i = 0; i < len; i++) {
	    if (getType(i) == Xml.TEXT) {
		buf.append(getText(i));
	    } else if (getType(i) == Xml.ELEMENT) {
		throw new RuntimeException("not text-only content!");
	    } 
	} 

	return buf.toString();
    } 

    /**
     * returns the text node with the given index or null if the node
     * with the given index is not a text node
     */
    public String getText(int index) {
	return getType(index) == Xml.TEXT ? (String) getChild(index) : null;
    } 

    /**
     * returns the type of the child at the given index. Possible
     * types are ELEMENT, TEXT, COMMENT, and PROCESSING_INSTRUCTION
     */
    public abstract int getType(int index);

    /**
     * returns the value of the given attribute. Convenience
     * method for getValue (getNamespace (), name
     */
    public String getValue(String name) {
	Attribute attr = getAttribute("", name);

	return attr == null ? null : attr.getValue();
    } 

    /**
     * returns the value of the given attribute
     */
    public String getValue(String namespace, String name) {
	Attribute attr = getAttribute(namespace, name);

	return attr == null ? null : attr.getValue();
    } 

    /**
     * @deprecated
     * please use indexOf (String name, int index
     */
    public int indexOf(String name) {
	return indexOf(getNamespace(), name, 0);
    } 

    /**
     * convenience method for indexOf (getNamespace (), name, startIndex)
     */
    public int indexOf(String name, int startIndex) {
	return indexOf(getNamespace(), name, startIndex);
    } 

    /**
     * convenience method for indexOf (namespace, name, 0)
     */
    public int indexOf(String namespace, String name) {
	return indexOf(namespace, name, 0);
    } 

    /**
     * searches for the given element name, starting at the given
     * start index. returns -1 if not found
     */
    public int indexOf(String namespace, String name, int startIndex) {
	int len = getChildCount();

	if (namespace == null) {
	    namespace = Xml.NO_NAMESPACE;
	} 

	for (int i = startIndex; i < len; i++) {
	    Element child = getElement(i);

	    if (child != null && namespace.equals(child.getNamespace()) 
		    && name.equals(child.getName())) {
		return i;
	    } 
	} 

	return -1;
    } 

    /**
     * by overwriting parse, the element can take complete control
     * over parsing its subtree
     */
    public void parse(Parser parser, StartTag startTag) throws IOException {
	if (startTag != null) {
	    setPrefixMap(startTag.getPrefixMap());
	} 

	boolean leave = false;

	do {
	    ParseEvent event = parser.read();

	    switch (event.getType()) {

	    case Xml.START_TAG:
		Element child = 
		    getDocument().createElement(this, event.getNamespace(), 
						event.getName(), 
						event.getAttributes());

		// System.out.println("Element.parse(): Adding a start child, name:" + child.getName());
		addChild(Xml.ELEMENT, child);
		child.parse(parser, (StartTag) event);

		break;

	    case Xml.TEXT:
		String txt = event.getText();

		// System.out.println("Element.parse(): Adding text child, name" + event.getText());
		addChild(Xml.TEXT, event.getText());

		break;

	    case Xml.COMMENT:

	    case Xml.PROCESSING_INSTRUCTION:

		// System.out.println("Element.parse(): Adding text child, name" + event.getText());
		addChild(event.getType(), event.getText());

		break;

	    case Xml.END_DOCUMENT:
		if (startTag != null) {
		    throw new RuntimeException("Unexpected end of document!");
		} 

	    case Xml.END_TAG:
		leave = true;

		break;

	    default:

		// System.out.println("Element.parse():Unexpected parse event: " + event);
		throw new RuntimeException("Unexpected parse event: " 
					   + event);
	    }
	} while (!leave);

	if (startTag != null &&!startTag.getDegenerated() 
		&& getChildCount() == 0) {
	    addChild(Xml.TEXT, "");
	} 
    } 

    /**
     * Method declaration
     *
     *
     * @param index
     *
     * @see
     */
    public abstract void removeAttribute(int index);

    /**
     * Method declaration
     *
     *
     * @param index
     *
     * @see
     */
    public abstract void removeChild(int index);

    /**
     * Method declaration
     *
     *
     * @param attr
     *
     * @see
     */
    public abstract void setAttribute(Attribute attr);

    /**
     * the setAttributes method is called during parsing
     */
    public abstract void setAttributes(Vector attributes);

    /**
     * sets the name of the element
     */
    public abstract void setName(String name);

    /**
     * sets the namespace of the element
     */
    public abstract void setNamespace(String namespace);

    /**
     * Sets the Parent of this element. Please use with care, you can
     * simply destroy the document tree structure using this method!
     */
    public abstract void setParent(Object parent);

    /**
     * returns a valid XML representation of this Element including
     * attributes and children
     */
    public String toString() {
	try {
	    ByteArrayOutputStream bos = new ByteArrayOutputStream();
	    XmlWriter		  xw = 
		new XmlWriter(new OutputStreamWriter(bos));

	    write(xw);
	    xw.close();

	    return new String(bos.toByteArray());
	} catch (IOException e) {
	    throw new RuntimeException("IO/Exception should not occur with a StringWriter!");
	} 
    } 

    /**
     * returns a valid XML representation of the children of this element
     */
    public String childrenToString() {
	try {
	    ByteArrayOutputStream bos = new ByteArrayOutputStream();
	    XmlWriter		  xw = 
		new XmlWriter(new OutputStreamWriter(bos));

	    write(xw);
	    xw.close();

	    return new String(bos.toByteArray());
	} catch (IOException e) {
	    throw new RuntimeException("IO/Exception should not occur with a StringWriter!");
	} 
    } 

    /**
     * Method declaration
     *
     *
     * @param writer
     *
     * @throws IOException
     *
     * @see
     */
    public void writeChildren(XmlWriter writer) throws IOException {
	int len = getChildCount();

	for (int i = 0; i < len; i++) {
	    switch (getType(i)) {

	    case Xml.ELEMENT:
		getElement(i).write(writer);

		break;

	    case Xml.TEXT:
		writer.write(getText(i));

		break;

	    default:
		writer.writeLegacy(getType(i), (String) getChild(i));
	    }
	} 
    } 

    /**
     * Method declaration
     *
     *
     * @param writer
     *
     * @throws IOException
     *
     * @see
     */
    public void writeStartTag(XmlWriter writer) throws IOException {
	Document doc = getDocument();
    } 

    /**
     * Method declaration
     *
     *
     * @param writer
     *
     * @throws IOException
     *
     * @see
     */
    public void write(XmlWriter writer) throws IOException {
	writer.writeStartTag(getPrefixMap(), getNamespace(), getName());

	int len = getAttributeCount();

	for (int i = 0; i < len; i++) {
	    Attribute attr = getAttribute(i);

	    writer.writeAttribute(attr.getNamespace(), attr.getName(), 
				  attr.getValue());
	} 

	writeChildren(writer);
	writer.writeEndTag();
    } 

}

